using System;
using System.Collections;
using System.Reflection;
using System.Drawing;

using Microsoft.DirectX;
using Microsoft.DirectX.DirectInput;

using DarkStrideToolbox;


namespace DarkStride.StellarLanes.SharedDLL
{
	public class Chassis : Item
	{
		#region Properties
		private Window_Popup m_oPopup = null;

		private int m_nCGridWidth = 0;
		private int m_nCGridHeight = 0;
		private int m_nMGridWidth = 0;
		private int m_nMGridHeight = 0;
		private enumChassisDirection m_nDirection = enumChassisDirection.Up;
		private int m_nXOffset = 0;
		private int m_nYOffset = 0;
		private DSSortedList m_oModules = new DSSortedList();
		private string m_sGUID = "";
		private int m_nDBSkinID = 0;
		private string m_sTexture = "";
		#endregion


		public Chassis()
		{
			this.PKeyObject = Globals.Inst().PrimaryKey.GetNewPrimarykey();
			//m_sGUID = DSMisc.GetGUID();
			m_nMGridWidth = 10;
			m_nMGridHeight = 10;
		}
		public Chassis( ComplexEntity oParentEntity )
		{
			this.PKeyObject = Globals.Inst().PrimaryKey.GetNewPrimarykey();
			//m_sGUID = DSMisc.GetGUID();
			m_nMGridWidth = 10;
			m_nMGridHeight = 10;
			this.ParentEntity = oParentEntity;
		}

		public virtual void AddModule( Module oModule )
		{
			oModule.Chassis = this;
			oModule.ParentEntity = this.ParentEntity;
			m_oModules.Add( oModule.PKey,oModule );
		}
		public virtual void DelModule( Module oModule )
		{
			oModule.UnAssign();
			oModule.Chassis = null;
			m_oModules.Remove( oModule.PKey );
			//Globals.Inst().Profile.UnassignedModules.Add( oModule.PKey,oModule );
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
			Module oLoopModule = null;


            base.Advance(oSession, nElapsedTime);

			/////////////////////////////////////////////
			//Normal
			/////////////////////////////////////////////
			//Advance our modules
			for( int i=0 ; i<m_oModules.Count ; i++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( i );
                oLoopModule.Advance(oSession, nElapsedTime);
			}
		}
		public virtual void PreRenderRenderSettingsChange( RenderSettings oRenderSettings )
		{
			Module oLoopModule = null;

			for( int i=0 ; i<m_oModules.Count ; i++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( i );
				oLoopModule.PreRenderRenderSettingsChange( oRenderSettings );
			}
		}
		public virtual void Render( RenderSettings oRenderSettings )
		{
			Module oLoopModule = null;


			//Render ourselves first
			if( m_sTexture.Length > 0 )
			{
				if( oRenderSettings.RenderType == enumRenderType.Underside && oRenderSettings.InGameScreen == enumGameScreen.Editor )
				{
					RenderChassis3D( m_sTexture,oRenderSettings,0 );
				}
				else if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
				{
					RenderChassis3D( m_sTexture,oRenderSettings,this.ParentEntity.Angle );
				}
			}

			//Now render our modules
			for( int i=0 ; i<m_oModules.Count ; i++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( i );
				oLoopModule.Render( oRenderSettings,this );
			}
		}
		public override void RenderForEditor3D( System.Drawing.Rectangle oAreaToDrawAt,RenderSettings oRenderSettings )
		{
			System.Drawing.Rectangle oRect = System.Drawing.Rectangle.Empty;
			Module oLoopModule = null;
			int nGridWidth = 0, nGridHeight = 0;


			//Render our base texture
			if( m_sTexture.Length > 0 &&
				(
					oRenderSettings.RenderType == enumRenderType.Topside ||
					( oRenderSettings.RenderType == enumRenderType.Underside && oRenderSettings.InGameScreen == enumGameScreen.Editor ) 
				)
			  )
			{
				if( oRenderSettings.RenderType == enumRenderType.Underside &&
					oRenderSettings.InGameScreen == enumGameScreen.Editor )
				{
					RenderOverlaySquares( oRenderSettings,oAreaToDrawAt,.50 );
				}

				Globals.Inst().GameEngine.RenderTexture2D( 
					m_sTexture,System.Drawing.Rectangle.Empty,
					oAreaToDrawAt,System.Drawing.Rectangle.Empty,Vector2.Empty,0,
					oRenderSettings.PercentTransparent,false, oRenderSettings.BaseDrawColor.ToArgb() );

				if( oRenderSettings.RenderType == enumRenderType.Topside &&
					oRenderSettings.InGameScreen == enumGameScreen.Editor )
				{
					RenderOverlaySquares( oRenderSettings,oAreaToDrawAt,.65 );
				}
			}
			else if( oRenderSettings.RenderType == enumRenderType.Modules )
			{
				System.Drawing.Color oPreModuleColor = oRenderSettings.BaseDrawColor;

				//If we are in the editor then the modules don't get the chassis color
				if( oRenderSettings.InGameScreen == enumGameScreen.Editor )
				{
					oRenderSettings.BaseDrawColor = System.Drawing.Color.White;
				}

				//Now render the modules
				nGridWidth = oAreaToDrawAt.Width / this.CGridWidth;
				nGridHeight = oAreaToDrawAt.Height / this.CGridHeight;
				for( int i=0 ; i<m_oModules.Count ; i++ )
				{
					oLoopModule = (Module)m_oModules.GetByIndex( i );

					oRect = new System.Drawing.Rectangle( 
						oAreaToDrawAt.X + nGridWidth * oLoopModule.XOffset,
						oAreaToDrawAt.Y - nGridHeight * oLoopModule.YOffset,
						nGridHeight * oLoopModule.MGridWidth,
						nGridHeight * oLoopModule.MGridHeight );
					oLoopModule.RenderForEditor3D( oRect,oRenderSettings );
				}

				oRenderSettings.BaseDrawColor = oPreModuleColor;
			}
		}
		public void RenderChassis3D( string sGraphicKey,RenderSettings oRenderSettings,double nAngle )
		{
			DSGobTable oTable = null;
			string sSkinTexture = "";
			string sBDmgTexture = "";
			Vector2 vScreenPtCenterShip = Vector2.Empty;
			Vector2 vWorldPtUpperLeftChassisCorner = Vector2.Empty;
			Vector2 vScreenPtUpperLeftChassisCorner = Vector2.Empty;
			Vector2 vRotate = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


			//This chassis upper left corner location
			vWorldPtUpperLeftChassisCorner = GetUpperLeftCornerWorldPt();
			vScreenPtUpperLeftChassisCorner = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vWorldPtUpperLeftChassisCorner );
			//Render location
			oRenderRect = new System.Drawing.Rectangle( 
				(int)( vScreenPtUpperLeftChassisCorner.X-1 ),
				(int)( vScreenPtUpperLeftChassisCorner.Y-1 ),
				(int)( m_nCGridWidth * GraphicConstants.m_cGRID_WIDTH * oRenderSettings.ZoomLevel + 2 ),
				(int)( m_nCGridHeight * GraphicConstants.m_cGRID_HEIGHT * oRenderSettings.ZoomLevel + 2 ) );

			//Get the center of my ship
			vScreenPtCenterShip = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,this.ParentEntity.Pos );

			//Where do we rotate around?  The center of the ship's difference from our upper left corner
			vRotate = new Vector2(	vScreenPtCenterShip.X - vScreenPtUpperLeftChassisCorner.X,
									vScreenPtCenterShip.Y - vScreenPtUpperLeftChassisCorner.Y );

			//Get our skin texture key
			if( m_nDBSkinID > 0 )
			{
				oTable = DSResourceManager.GetGlobalInstance().GetGobTable( GobConstants.m_cTABLE_SKINS );
				sSkinTexture = (string)oTable.FindValue( GobConstants.m_cCLMN_SKINID,m_nDBSkinID,GobConstants.m_cCLMN_SKINTEXTURENAME );
			}

			//Do we show battle damage?  For now, always.
			sBDmgTexture = GetBDmgTextureKey();

			/*//Now, render it.
			Globals.Inst().RenderTextureWithBattleDamage(
				sGraphicKey,System.Drawing.Rectangle.Empty,
				sSkinTexture,System.Drawing.Rectangle.Empty,
				sBDmgTexture,System.Drawing.Rectangle.Empty,
				oRenderRect,System.Drawing.Rectangle.Empty,vRotate,nAngle,oRenderSettings.PercentTransparent,
				oRenderSettings.BaseDrawColor.ToArgb() );*/

			Globals.Inst().GameEngine.RenderTexture2D( 
							sGraphicKey,System.Drawing.Rectangle.Empty,
							oRenderRect,System.Drawing.Rectangle.Empty,vRotate,nAngle,
							oRenderSettings.PercentTransparent,false, oRenderSettings.BaseDrawColor.ToArgb() );
		}
		public void RenderOverlaySquares( RenderSettings oRenderSettings,System.Drawing.Rectangle oAreaToDrawAt,double nTransparency )
		{
			double nGridXSize = 0;
			double nGridYSize = 0;
			System.Drawing.Rectangle oRect = System.Drawing.Rectangle.Empty;
			string sTexture = string.Empty;


			//Get the square sizes
			nGridXSize = (double)oAreaToDrawAt.Width / (double)this.CGridWidth;
			nGridYSize = (double)oAreaToDrawAt.Height / (double)this.CGridHeight;

			//Render the transparent squares
			for( int nX=0 ; nX<this.CGridWidth ; nX++ )
			{
				for( int nY=0 ; nY>-this.CGridHeight ; nY-- )
				{
					oRect = new System.Drawing.Rectangle( 
						(int)( nX * nGridXSize + oAreaToDrawAt.X ),
						(int)( nY * -nGridYSize + oAreaToDrawAt.Y ),
						(int)nGridXSize,(int)nGridYSize );

					if( oRenderSettings.EditorDisplay != enumEditorObject.Unassigned && 
						this.CGridIsInUse( nX,nY ) == true )
					{
						sTexture = GraphicConstants.m_cGRID_GREY;
						if( this.MGridModuleTypeAllowed( nX,nY ) == SharedDLL.enumModuleType.Weapon )
						{
							sTexture = GraphicConstants.m_cGRID_RED;
						}
						else if( this.MGridModuleTypeAllowed( nX,nY ) == SharedDLL.enumModuleType.Engine )
						{
							sTexture = GraphicConstants.m_cGRID_GREEN;
						}
						else if( this.MGridModuleTypeAllowed( nX,nY ) == SharedDLL.enumModuleType.PowerGen )
						{
							sTexture = GraphicConstants.m_cGRID_YELLOW;
						}
						else if( this.MGridModuleTypeAllowed( nX,nY ) == SharedDLL.enumModuleType.Other )
						{
							sTexture = GraphicConstants.m_cGRID_PURPLE;
						}
						else if( this.MGridModuleTypeAllowed( nX,nY ) == SharedDLL.enumModuleType.ShieldGen )
						{
							//sTexture = GraphicConstants.m_cGRID_BLUE;
						}

						Globals.Inst().GameEngine.RenderTexture2D( sTexture,
							System.Drawing.Rectangle.Empty,oRect,Vector2.Empty,
							0,nTransparency,false,System.Drawing.Color.White.ToArgb() );
					}
					if( this.CGridIsConnector( /*this.XOffset+*/nX,/*this.YOffset+*/nY ) == true )
					{
						Globals.Inst().GameEngine.RenderTexture2D( 
							GraphicConstants.m_cGRID_BLUE,
							System.Drawing.Rectangle.Empty,oRect,Vector2.Empty,
							0,.6,false,System.Drawing.Color.White.ToArgb() );
					}
				}
			}
		}

		public virtual Vector2 GetUpperLeftCornerWorldPt()
		{
			Vector2 vWorldPtBaseChassisUpperLeftCorner = Vector2.Empty;
			Vector2 vWorldPtUpperLeftChassisCorner = Vector2.Empty;


			//The center point is actually the center of the base chassis
			vWorldPtBaseChassisUpperLeftCorner = new Vector2(
				(float)( this.ParentEntity.Pos.X - Math.Floor( this.ParentEntity.BaseChassis.CGridWidth * GraphicConstants.m_cGRID_WIDTH / 2.0 ) ),
				(float)( this.ParentEntity.Pos.Y - Math.Floor( this.ParentEntity.BaseChassis.CGridHeight * GraphicConstants.m_cGRID_HEIGHT / 2.0 ) ) );

			//This chassis upper left corner location
			vWorldPtUpperLeftChassisCorner = new Vector2( 
				vWorldPtBaseChassisUpperLeftCorner.X + 
				( m_nXOffset - this.ParentEntity.BaseChassis.XOffset ) * GraphicConstants.m_cGRID_WIDTH,
				vWorldPtBaseChassisUpperLeftCorner.Y + 
				( this.ParentEntity.BaseChassis.YOffset - m_nYOffset ) * GraphicConstants.m_cGRID_HEIGHT );


			return( vWorldPtUpperLeftChassisCorner );
		}
		public virtual Vector2 GetCenterWorldPt()
		{
			Vector2 vWorldPtCenter = Vector2.Empty;
			Vector2 vWorldPtUpperLeftChassisCorner = GetUpperLeftCornerWorldPt();


			vWorldPtCenter = new Vector2( 
				(float)( vWorldPtUpperLeftChassisCorner.X + ( m_nCGridWidth / 2.0 ) * GraphicConstants.m_cGRID_WIDTH ),
				(float)( vWorldPtUpperLeftChassisCorner.Y + ( m_nCGridHeight / 2.0 ) * GraphicConstants.m_cGRID_HEIGHT ) );


			return( vWorldPtCenter );
		}
		public virtual RegionCircle GetCircleRegion( double nRadius )
		{
			Vector2 vCenterWorldPt = Vector2.Empty;
			Vector2 vRotatedCenterWorldPt = Vector2.Empty;
			RegionCircle oRegion = null;


			vCenterWorldPt = GetCenterWorldPt();
			vRotatedCenterWorldPt = this.ParentEntity.GetRotatedWorldPt( vCenterWorldPt );

			oRegion = new RegionCircle( vRotatedCenterWorldPt,(float)nRadius );


			return( oRegion );
		}

		private string GetBDmgTextureKey()
		{
			string sBDmgTexture = "";

			if( (double)this.StructurePoints / (double)this.MaxStructurePoints < .75 )
			{
				sBDmgTexture = "Skin_Damage2";
			}

			return( sBDmgTexture );
		}

		public override void HoverStart()
		{
			m_oPopup = new Window_Popup();
			m_oPopup.Display( Globals.Inst().GameEngine.MouseCursor,
							  this,m_sTexture,this.GetHoverRenderText() );
		}
		public override void HoverStop()
		{
			if( m_oPopup != null )
			{
				m_oPopup.Dispose();
				m_oPopup = null;
			}
		}
		public override ArrayList GetHoverRenderText()
		{
			ArrayList oProperties = base.GetHoverRenderText();
			double nPercRedux = 0;
			double nFixedRedux = 0;
			double nMass = 0;

		
			//Build the string
			oProperties.Add( "Structure: " + this.StructurePoints.ToString( "0" ) + " / " + this.MaxStructurePoints.ToString( "0" ) );

			nPercRedux = this.GetPropertyTotal( enumEntProperties.Structure_PercentDamageRedux,true );
			if( nPercRedux != 0 )
			{ 
				oProperties.Add( "Damage Resist: " + nPercRedux.ToString( "0.00" ) + "%" );
			}

			nFixedRedux = this.GetPropertyTotal( enumEntProperties.Structure_FixedDamageRedux,true );
			if( nFixedRedux != 0 )
			{ 
				oProperties.Add( "Damage Resist: " + nFixedRedux.ToString( "0.00" ) );
			}

			nMass = this.GetPropertyTotal( enumEntProperties.Structure_Mass,true );
			if( nMass != 0 )
			{ 
				oProperties.Add( "Mass: " + nMass.ToString( "0" ) + " tons" );
			}

			return( oProperties );
		}

        public virtual void KeyProcess(Session oSession, Microsoft.DirectX.DirectInput.Key oKey, bool bPressed)
		{
			Module oLoopModule = null;

			for( int n=0 ; n<m_oModules.Count ; n++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( n );
                oLoopModule.KeyProcess(oSession,oKey, bPressed);
			}
		}

        public virtual void NetworkMessageReceived(Session oSession, long nMessageType, string sMessage)
		{
			Module oLoopModule = null;

			for( int n=0 ; n<m_oModules.Count ; n++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( n );
                oLoopModule.NetworkMessageReceived(oSession,nMessageType, sMessage);
			}
		}

		public override DSSerialize Serialize(Session oSession)
		{
			Module oLoopModule = null;
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
			oSerialize.Set( 0,m_nCGridWidth );
			oSerialize.Set( 1,m_nCGridHeight );
			oSerialize.Set( 2,Convert.ToInt32( m_nDirection ) );
			oSerialize.Set( 3,m_nXOffset );
			oSerialize.Set( 4,m_nYOffset );
			oSerialize.Set( 5,this.PKeyObject.Serialize() );
			oSerialize.SetList( 6 );
			
			for( int i=0 ; i<m_oModules.Count ; i++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( i );

                if (oLoopModule.GetType() != typeof(GenericModule))
                {
                    oSerialize.SetListItem(6, oLoopModule.GetType().Name);
                }
                else
                {
                    oSerialize.SetListItem(6, null);
                }
                oSerialize.SetListItem(6, oLoopModule.Serialize(oSession));
			}
			oSerialize.Set( 7,m_nMGridWidth );
			oSerialize.Set( 8,m_nMGridHeight );
            oSerialize.Set(9, base.Serialize(oSession));
			oSerialize.Set( 10,m_nDBSkinID );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
            string sTemp = string.Empty;
			Assembly oControlAssembly = null;
			Type oControlType = null;
			ArrayList oModules = null;
			Module oNewModule = null;
			Module oLoopModule = null;
			DSSortedList oModulesNotUsedYet = null;


			oModulesNotUsedYet = m_oModules.Clone();
			m_oModules.Clear();


			m_nCGridWidth = oSerialize.GetInt( 0 );
			m_nCGridHeight = oSerialize.GetInt( 1 );
			m_nDirection = (enumChassisDirection)oSerialize.GetInt( 2 );
			m_nXOffset = oSerialize.GetInt( 3 );
			m_nYOffset = oSerialize.GetInt( 4 );
			this.PKeyObject.DeSerialize( (DSSerialize)oSerialize.Get( 5 ) );
			oModules = oSerialize.GetList( 6 );

			//Walk the entitys and update them
			for( int i=0 ; i<oModules.Count ; i+=2 )
			{
				//Deserialize the form into a new object... so first create the new object
				oControlAssembly = Assembly.Load("StellarLanes");//.LoadWithPartialName( "StellarLanes" );//(string)oModules[ i+1 ] );
                if (oModules[i] == null)
                {
                    sTemp = typeof( GenericModule ).Name;
                }
                else
                {
                    sTemp = (string)oModules[i];
                }
                oControlType = oControlAssembly.GetType("DarkStride.StellarLanes.SharedDLL." + sTemp);
				//Make it!
				oNewModule = (Module)Activator.CreateInstance( oControlType,new object[]{ this.ParentEntity } );
				if( oNewModule.Active == true )
				{
					//Now deserialize the form
					oNewModule.DeSerialize( oSession,(DSSerialize)oModules[ i+1 ] );


					//Do we have this module?
					oLoopModule = (Module)oModulesNotUsedYet.GetByKey( oNewModule.PKey );
					if( oLoopModule == null )
					{
						AddModule( oNewModule );
					}
					//If we found it then update it, we can do this outside the lock
					if( oLoopModule != null )
					{
						oLoopModule.DeSerialize( oSession,(DSSerialize)oModules[ i+1 ] );
						AddModule( oLoopModule );
						oModulesNotUsedYet.Remove( oLoopModule.PKey );
					}
				}
			}


			//All done?  Any spare modules?  If so they go into unassigned.
            if (Globals.Inst().Profile != null && Globals.Inst().OurShip == this.ParentEntity )
			{
				for( int i=0 ; i<oModulesNotUsedYet.Count ; i++ )
				{
					oLoopModule = (Module)oModulesNotUsedYet.GetByIndex( i );
					oLoopModule.UnAssign();
                    Globals.Inst().Profile.AddModule( oLoopModule );
				}
			}

			m_nMGridWidth = oSerialize.GetInt( 7 );
			m_nMGridHeight = oSerialize.GetInt( 8 );
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 9 ) );
			m_nDBSkinID = oSerialize.GetInt( 10 );
		}

		public virtual void Assign()
		{
		}
		public virtual void UnAssign()
		{
			Module oLoopModule = null;


			try
			{
				for( int i=0 ; i<m_oModules.Count ; i++ )
				{
					oLoopModule = (Module)m_oModules.GetByIndex( i );
					if( Globals.Inst().Profile != null )
					{
						Globals.Inst().Profile.AddModule( oLoopModule );
					}
					DelModule( oLoopModule );
				}
			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors( oEx );
			}
		}
       
		public bool CGridIsInUse( int nXOffset,int nYOffset )
		{
			bool bGridInUse = false;
			enumModuleType nType = MGridModuleTypeAllowed( nXOffset /*- m_nXOffset*/, nYOffset /*- m_nYOffset*/ );
			bGridInUse = ( (int)nType & (int)enumModuleType.OpenSpace ) != (int)enumModuleType.OpenSpace;
			//bGridInUse = ( nType != enumModuleType.OpenSpace );
			return( bGridInUse );
		}
		public bool CGridIsConnector( int nXOffset,int nYOffset )
		{
			int nType = (int)MGridModuleTypeAllowed( nXOffset, nYOffset );
			return( ( nType & (int)enumModuleType.Connector ) == (int)enumModuleType.Connector );
		}		
		public virtual enumModuleType MGridModuleTypeAllowed( int nXOffset,int nYOffset )
		{
			if( MGridIsInUse( nXOffset,nYOffset ) == true )
			{
				return( enumModuleType.All );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}			
		}

		public virtual ArrayList GetCGridConnectors()
		{
			System.Drawing.Point oConnector = System.Drawing.Point.Empty;
			ArrayList oaConnectors = new ArrayList();


			for( int nX=0 ; nX<m_nCGridWidth ; nX++ )
			{
				for( int nY=0 ; nY>-m_nCGridHeight ; nY-- )
				{
					if( CGridIsConnector( nX,nY ) == true )
					{
						oConnector = new System.Drawing.Point( m_nXOffset+nX,m_nYOffset+nY );
						oaConnectors.Add( oConnector );
					}
				}
			}


			return( oaConnectors );
		}
		public Module GetModuleAtMGridLoc( int nMXOffset,int nMYOffset )
		{
			Module oModuleFound = null;
			Module oLoopModule = null;


			for( int m=0 ; m<m_oModules.Count ; m++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( m );
				if( oLoopModule.MGridIsInUse( nMXOffset-oLoopModule.XOffset,nMYOffset-oLoopModule.YOffset ) == true )
				{
					oModuleFound = oLoopModule;
					break;
				}
			}


			return( oModuleFound );
		}
		public virtual bool MGridIsInUse( int nMXOffset,int nMYOffset )
		{
			Module oLoopModule = null;
			bool bInUse = false;


			for( int i=0 ; i<m_oModules.Count ; i++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( i );
				if( oLoopModule.MGridIsInUse( nMXOffset-oLoopModule.XOffset,nMYOffset-oLoopModule.YOffset ) == true )
				{
					bInUse = true;
					break;
				}
			}

			
			return( bInUse );
		}
		public virtual bool CanModuleBePlaced( Module oModule,int nMXOffset,int nMYOffset,ref ArrayList oaConflicts )
		{
			int nModuleTypeNeeded = 0;
			int nModuleTypeFound = 0;
			bool bCanBePlacedHere = true;
			Point oConflict = Point.Empty;


			oaConflicts = new ArrayList();

			for( int nX=0 ; nX<oModule.MGridWidth ; nX++ )
			{
				for( int nY=0 ; nY>-oModule.MGridHeight ; nY-- )
				{
					nModuleTypeNeeded = (int)oModule.MGridTypeNeeded( nX,nY );
					nModuleTypeFound = (int)MGridModuleTypeAllowed( nMXOffset+nX,nMYOffset+nY );

					if( oModule.MGridIsInUse( nX,nY ) == true &&
						nModuleTypeNeeded != (int)enumModuleType.CantBeUsed &&
						(
							MGridIsInUse( nMXOffset+nX,nMYOffset+nY ) == true || 
							( ( nModuleTypeFound & nModuleTypeNeeded ) != nModuleTypeNeeded )
						)
					  )
					{
						oConflict = new Point( nX,nY );
						oaConflicts.Add( oConflict );
						bCanBePlacedHere = false;
					}
				}
			}


			return( bCanBePlacedHere );
		}
		public virtual ArrayList GetRegions()
		{
			return( new ArrayList() );
		}
		public virtual Region HitTest( Region oTestRegion )
		{
			Region oRegion = null;
			Region oHitTestRegion = null;
			Region oRetVal = new Region();
			ArrayList oRegions = GetRegions();			

			for( int i=0 ; i<oRegions.Count ; i++ )
			{
				oRegion = (Region)oRegions[i];
				oHitTestRegion = oRegion.HitTest( oTestRegion );
				if( Region.IsEmpty( oHitTestRegion ) == false )
				{
					oRetVal = oHitTestRegion;
					break;
				}
			}
			return( oRetVal );
		}

		public virtual double DealDamage( Session oSession,double nAmount,Region oDamageWhere )
		{
			bool bRemoveChassis = false;
			double nActualDamage = nAmount;
			double nDmgLeftToDistribute = 0;
			double nTempDamage = 0;
			double nFixedRedux = 0;
			double nPercRedux = 0;
			Module oLoopModule = null;
			DSNetworkPacket oPacket = null;
			ComplexEntity oParentEntity = null;


			try
			{
				//Start by damaging this chassis
				nFixedRedux = this.GetPropertyTotal( enumEntProperties.Structure_FixedDamageRedux,false );
				nPercRedux = this.GetPropertyTotal( enumEntProperties.Structure_PercentDamageRedux,false );
				nActualDamage = base.ReduceDamage( nAmount,nFixedRedux,nPercRedux );
				base.StructurePoints -= nActualDamage;
				base.StructurePoints = DSMisc.Max( 0,base.StructurePoints );

				//Now damage our modules
				nDmgLeftToDistribute = nActualDamage;
				for( int i=0 ; i<m_oModules.Count && nDmgLeftToDistribute>0 ; i++ )
				{
					oLoopModule = (Module)m_oModules.GetByIndex( i );

					//Do we damage this one?
					if( DSMisc.GetRnd() < .5 )
					{
						nTempDamage = DSMisc.GetRnd() * ( nDmgLeftToDistribute / m_oModules.Count );
						nDmgLeftToDistribute -= nTempDamage;
						oLoopModule.DealDamage( nTempDamage,oDamageWhere );
					}
				}

				//11/05/2007 Chris Hill  Make sure to accrue damage before you potentially remove the chassis.
				this.ParentEntity.AccrueDamageRollingText( oDamageWhere.ConvertToPointLocation(),nActualDamage,nAmount );

				//Lets do fancy stuff... can we 'loose' this chassis?
				if( base.StructurePoints < 0 && Globals.Inst().IAmTheServer == true && this.ParentEntity != null && 
					this.ParentEntity.GetDependentChildrenOfChassis( this ).Count == 0 )
				{
					//Spit it off into its own entity
					bRemoveChassis = false;
					oParentEntity = this.ParentEntity;
					if( DSMisc.GetRnd() < .1 )
					{				
						bRemoveChassis = true;
						SpinOffLostPeice(oSession);
					}
					else if( DSMisc.GetRnd() < .2 )
					{
						bRemoveChassis = true;
						oParentEntity.RemoveChassis( this );
					}

					if( bRemoveChassis == true )
					{
						oPacket = NetMsg.Pckt_EntityRemoveChassis( oParentEntity,this );
                        oSession.SendMsgOnToAllPlayersInSession(oPacket);
					}
				}
			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors( oEx );
			}

			 return( nDmgLeftToDistribute );
		}
		private void SpinOffLostPeice(Session oSession)
		{
			Vector2 vWorldPtCenter = Vector2.Empty;
			Vector2 vRotatedWorldPtCenter = Vector2.Empty;
			ComplexEntity oCarrierComplexEntity = null;
            Zone oMyZone = null;


			//Create our carrier entity
			oCarrierComplexEntity = new ComplexEntity();

			//Set its defaults
			vWorldPtCenter = GetCenterWorldPt();
			vRotatedWorldPtCenter = this.ParentEntity.GetRotatedWorldPt( vWorldPtCenter );
			//Go!
			oCarrierComplexEntity.OwnerSocketID = -1;
			oCarrierComplexEntity.CanPickUpItems = false;
			oCarrierComplexEntity.Pos = vRotatedWorldPtCenter;
			oCarrierComplexEntity.Vel = this.ParentEntity.Vel;
			oCarrierComplexEntity.Angle = this.ParentEntity.Angle;
			oCarrierComplexEntity.AngularMomentum = DSMisc.GetRnd() * 6 - 3;

			//Remove this chassis from our current entity
			this.ParentEntity.RemoveChassis( this );

			//Add it to the world
            oMyZone = oSession.Zones[this.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];
			oCarrierComplexEntity.AddChassis( this );
            oMyZone.AddEntity(oSession,oCarrierComplexEntity, true);
		}
		public virtual void Collision(Entity oCollidedWith, double nElapsedTime, Region oCollisionRegion)
		{
		}

		public override double GetPropertyTotal( enumEntProperties nPropertyToGet,bool bIncludeChildren )
		{
			Module oLoopModule = null;
			double nPropertyTotal = 0;

			nPropertyTotal = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );

			if( bIncludeChildren == true )
			{
				for( int n=0 ; n<m_oModules.Count ; n++ )
				{
					oLoopModule = (Module)m_oModules.GetByIndex( n );
					if( nPropertyToGet == enumEntProperties.HUD_HPDetail )
					{
						nPropertyTotal = DSMisc.Max( nPropertyTotal,
							oLoopModule.GetPropertyTotal( nPropertyToGet,bIncludeChildren ) );
					}
					else
					{
						nPropertyTotal += oLoopModule.GetPropertyTotal( nPropertyToGet,bIncludeChildren );
					}
				}
			}

			return( nPropertyTotal );
		}
		public override double TakeEnergy( double nAmountToTake )
		{
			Module oLoopModule = null;
			double nEnergyFound = 0;


			nEnergyFound = base.TakeEnergy( nAmountToTake );

			for( int n=0 ; n<m_oModules.Count ; n++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( n );

				nEnergyFound += oLoopModule.TakeEnergy( nAmountToTake - nEnergyFound );
			}


			return( nEnergyFound );
		}

		public void RestoreToFullHealth()
		{
			Module oLoopModule = null;

			this.StructurePoints = this.MaxStructurePoints;

			for( int n=0 ; n<m_oModules.Count ; n++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( n );
				oLoopModule.StructurePoints = oLoopModule.MaxStructurePoints;
			}
		}
		public virtual double GetPreArmorDamageReduction( double nDamage )
		{
			Module oLoopModule = null;
			double nNewDamage = nDamage;

			for( int n=0 ; n<m_oModules.Count ; n++ )
			{
				oLoopModule = (Module)m_oModules.GetByIndex( n );
				nNewDamage = oLoopModule.GetPreArmorDamageReduction( nNewDamage );
			}

			return( nNewDamage );
		}


		#region Properties
		public int XOffset
		{
			get
			{
				return( m_nXOffset );
			}
			set
			{
				m_nXOffset = value;
			}
		}
		public int YOffset
		{
			get
			{
				return( m_nYOffset );
			}
			set
			{
				m_nYOffset = value;
			}
		}
		public DSSortedList Modules
		{
			get
			{
				return( m_oModules );
			}
			set
			{
				m_oModules = value;
			}
		}
		public string GUID_Obsolete
		{
			get
			{
				return( m_sGUID );
			}
			set
			{
				m_sGUID = value;
			}
		}
		public virtual int CGridWidth 
		{
			get
			{
				return( m_nCGridWidth );
			}
			set
			{
				m_nCGridWidth = value;
			}
		}
		public virtual int CGridHeight
		{
			get
			{
				return( m_nCGridHeight );
			}
			set
			{
				m_nCGridHeight = value;
			}
		}
		public virtual int MGridWidth 
		{
			get
			{
				return( m_nMGridWidth );
			}
			set
			{
				m_nMGridWidth = value;
			}
		}
		public virtual int MGridHeight
		{
			get
			{
				return( m_nMGridHeight );
			}
			set
			{
				m_nMGridHeight = value;
			}
		}
		public virtual enumChassisDirection Direction
		{
			get
			{
				return( m_nDirection );
			}
			set
			{
				m_nDirection = value;
			}
		}
		public int DBSkinID
		{
			get
			{
				return( m_nDBSkinID );
			}
			set
			{
				m_nDBSkinID = value;
			}
		}
		public string Texture
		{
			get
			{
				return( m_sTexture );
			}
			set
			{
				m_sTexture = value;
			}
		}
		public override ComplexEntity ParentEntity
		{
			get
			{
				return( base.ParentEntity );
			}
			set
			{
				Module oLoopModule = null;

				for( int i=0 ; i<m_oModules.Count ; i++ )
				{
					oLoopModule = (Module)m_oModules.GetByIndex( i );
					oLoopModule.ParentEntity = value;
				}

				base.ParentEntity = value;
			}
		}
		#endregion
	}


	public class GenericChassis : Chassis
	{
		#region Properties
		private int m_nChassisID = -1;
		private int[,] m_naModuleTypes = null;
		private string m_sName = "";
		#endregion

		public GenericChassis()
		{
			//Do not put code here
			throw new System.Exception( "Initializing chassis with wrong constructor." );
		}
		public GenericChassis( ComplexEntity oParentEntity)
		{
			base.ParentEntity = oParentEntity;
		}

		public GenericChassis( ComplexEntity oParentEntity,int nChassisID )
		{
			m_nChassisID = nChassisID;
			base.ParentEntity = oParentEntity;
		}


		public override void CreatedAsNewItem()
		{
			string sDifficultyKey = "";
			string sModuleTypes = "";
			string[] saModules = null;
			string[] saFields = null;
			int nRow = 0;
			int nX = 0, nY = 0, nType = 0;
			DSGobTable oChassisTable = null;


			base.CreatedAsNewItem();

			if( m_nChassisID != -1 )
			{
				oChassisTable = DSResourceManager.GetGlobalInstance().GetGobTable( GobConstants.m_cTABLE_CHASSIS );
				//Find the row for this ID
				nRow = oChassisTable.FindRow( "GenericChassisID",m_nChassisID );
		
				if( this.ItemLevel == enumItemLevel.Normal )	{ sDifficultyKey = "_Nrm"; }
				else if( this.ItemLevel == enumItemLevel.Advanced )	{ sDifficultyKey = "_Adv"; }
				else if( this.ItemLevel == enumItemLevel.Expert )	{ sDifficultyKey = "_Exp"; }

				//Get the rendering components
				base.Texture = DSMisc.TypeUtils.GetSafeStr( oChassisTable.GetData( "Texture",nRow ) );
				base.CGridWidth = DSMisc.TypeUtils.GetSafeInt( oChassisTable.GetData( "Width",nRow ) );
				base.MGridWidth = base.CGridWidth;
				base.CGridHeight = DSMisc.TypeUtils.GetSafeInt( oChassisTable.GetData( "Height",nRow ) );
				base.MGridHeight = base.CGridHeight;

				//Get the Difficulty dependents
				m_sName = DSMisc.TypeUtils.GetSafeStr( oChassisTable.GetData( "Name" + sDifficultyKey,nRow ) );
				this.MinLevel = DSMisc.TypeUtils.GetSafeLng( oChassisTable.GetData( "MinLevel" + sDifficultyKey,nRow ) );
				base.StructurePoints	= DSMisc.TypeUtils.GetSafeLng( oChassisTable.GetData( "StructurePoints" + sDifficultyKey,nRow ) );
				base.MaxStructurePoints = base.StructurePoints;
				base.PercentDamageRedux = DSMisc.TypeUtils.GetSafeLng( oChassisTable.GetData( "PercentDamageRedux" + sDifficultyKey,nRow ) );
				base.FixedDamageRedux	= DSMisc.TypeUtils.GetSafeLng( oChassisTable.GetData( "FixedDamageRedux" + sDifficultyKey,nRow ) );
				base.Mass				= DSMisc.TypeUtils.GetSafeDbl( oChassisTable.GetData( "Mass" + sDifficultyKey,nRow ) );			

				//Now load in our grid layouts
				sModuleTypes = DSMisc.TypeUtils.GetSafeStr( oChassisTable.GetData( "ModuleType",nRow ) );
				m_naModuleTypes = new int[ base.CGridWidth,base.CGridHeight ];
				saModules = DSMisc.Split( sModuleTypes ,";" );
				for( int i=0 ; i<saModules.Length-1 ; i++ )
				{
					try
					{
						saFields = DSMisc.Split( saModules[ i ],"," );
						nX		= DSMisc.TypeUtils.GetSafeInt( saFields[0] );
						nY		= DSMisc.TypeUtils.GetSafeInt( saFields[1] );
						nType	= DSMisc.TypeUtils.GetSafeInt( saFields[2] );

						if( nX >= 0 && nX < base.CGridWidth && nY >= 0 && nY < base.CGridHeight )
						{
							m_naModuleTypes[ nX,nY ] = nType;
						}
					}
					catch( System.Exception oEx )
					{
						DSMisc.ShowErrors( oEx );
					}
				}
			}
		}


		public override enumModuleType MGridModuleTypeAllowed( int nXOffset,int nYOffset )
		{
			enumModuleType nRetVal = enumModuleType.OpenSpace;

			if( nXOffset >= 0 && nXOffset < base.CGridWidth && nYOffset <= 0 && nYOffset > -base.CGridHeight )
			{
				nRetVal = (enumModuleType)m_naModuleTypes[ nXOffset,-nYOffset ];
			}

			return( nRetVal );
		}

		public override ArrayList GetRegions()
		{
			Vector2 vWorldPtCenter = GetCenterWorldPt();
			Vector2 vRotatedWorldPt = base.ParentEntity.GetRotatedWorldPt( vWorldPtCenter );

			RegionCircle oMe = new RegionCircle( vRotatedWorldPt,20 );
			ArrayList oList = new ArrayList();
			oList.Add( oMe );
			return( oList );
		}


		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,base.Texture );
			oSerialize.Set( 2,base.CGridWidth );
			oSerialize.Set( 3,base.CGridHeight );
			oSerialize.Set( 4,m_sName );
			oSerialize.SetList( 5 );

			for( int nX=0 ; nX<base.CGridWidth ; nX++ )
			{
				for( int nY=0 ; nY<base.CGridHeight ; nY++ )
				{
					if( m_naModuleTypes[ nX,nY ] != 0 )
					{
						oSerialize.SetListItem( 5,nX );
						oSerialize.SetListItem( 5,nY );
						oSerialize.SetListItem( 5,m_naModuleTypes[ nX,nY ] );
					}
				}
			}


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			ArrayList oArrayList = null;

			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			base.Texture		= oSerialize.GetString( 1 );
			base.CGridWidth		= oSerialize.GetInt( 2 );

if( base.CGridWidth == 0 )
{
	throw new System.Exception( "Can't be zero!" );
}

			base.CGridHeight	= oSerialize.GetInt( 3 );
			m_sName				= oSerialize.GetString( 4 );

			oArrayList = oSerialize.GetList( 5 );
			m_naModuleTypes = new int[ base.CGridWidth,base.CGridHeight ];
			for( int i=0 ; i<oArrayList.Count ; i+=3 )
			{
				m_naModuleTypes[ Convert.ToInt32( oArrayList[ i ] ),
								 Convert.ToInt32( oArrayList[ i+1 ] ) ] = Convert.ToInt32( oArrayList[ i+2 ] );
			}
		}

        
		#region Properties
		public override string Name
		{
			get
			{
				return( m_sName );
			}
		}
		#endregion
	}
}